# Sample Terraform: cross-account IAM role for Xshield Cloud Connector (AWS).
# Use with the ColorTokens account ID provided in the Xshield portal.
# See README.md for instructions.

terraform {
  required_version = ">= 1.0"
  required_providers {
    aws = {
      source  = "hashicorp/aws"
      version = ">= 4.0"
    }
  }
}

resource "aws_iam_role" "colortokens_cross_account" {
  name               = var.role_name
  description        = "Allows ColorTokens (Xshield) to assume this role for read-only discovery and monitoring"
  assume_role_policy = data.aws_iam_policy_document.trust_colortokens.json
  tags               = var.tags
}

data "aws_iam_policy_document" "trust_colortokens" {
  statement {
    sid     = "AllowColorTokensAssumeRole"
    effect  = "Allow"
    actions = ["sts:AssumeRole"]
    principals {
      type        = "AWS"
      identifiers = ["arn:aws:iam::${var.colortokens_account_id}:role/ColorTokensCloudSecurityServiceRole"]
    }
  }
}

resource "aws_iam_role_policy" "colortokens_access" {
  name   = "ColorTokensAccessPolicy"
  role   = aws_iam_role.colortokens_cross_account.id
  policy = data.aws_iam_policy_document.colortokens_access.json
}

data "aws_iam_policy_document" "colortokens_access" {
  # S3
  statement {
    sid       = "ColorTokensS3FullReadAccess"
    effect    = "Allow"
    actions   = ["s3:ListAllMyBuckets", "s3:GetBucketLocation", "s3:GetBucketAcl", "s3:GetBucketPolicy", "s3:GetBucketVersioning", "s3:GetBucketPublicAccessBlock", "s3:GetBucketNotification", "s3:GetBucketTagging", "s3:ListBucket", "s3:GetObject"]
    resources = ["*"]
  }
  # Lambda
  statement {
    sid       = "ColorTokensLambdaReadAccess"
    effect    = "Allow"
    actions   = ["lambda:ListFunctions", "lambda:GetFunction", "lambda:GetFunctionConfiguration", "lambda:GetPolicy", "lambda:ListTags", "lambda:ListEventSourceMappings"]
    resources = ["*"]
  }
  # DynamoDB
  statement {
    sid       = "ColorTokensDynamoReadAccess"
    effect    = "Allow"
    actions   = ["dynamodb:ListTables", "dynamodb:DescribeTable", "dynamodb:ListTagsOfResource"]
    resources = ["*"]
  }
  # EC2 (instances, networking, flow logs)
  statement {
    sid       = "ColorTokensEC2ReadAccess"
    effect    = "Allow"
    actions   = ["ec2:DescribeInstances", "ec2:DescribeTags", "ec2:DescribeSecurityGroups", "ec2:DescribeNetworkAcls", "ec2:DescribeSubnets", "ec2:DescribeVpcs", "ec2:DescribeVolumes", "ec2:DescribeSnapshots", "ec2:DescribeAddresses", "ec2:DescribeNetworkInterfaces", "ec2:DescribeRegions", "ec2:DescribeAvailabilityZones", "ec2:DescribeFlowLogs"]
    resources = ["*"]
  }
  # EC2 (NAT, gateways, VPC endpoints)
  statement {
    sid       = "ColorTokensNetworkResourceReadAccess"
    effect    = "Allow"
    actions   = ["ec2:DescribeNatGateways", "ec2:DescribeInternetGateways", "ec2:DescribeTransitGateways", "ec2:DescribeVpcEndpoints", "ec2:DescribeVpcEndpointServices"]
    resources = ["*"]
  }
  # EC2 (prefix lists)
  statement {
    sid       = "ColorTokensPrefixListReadAccess"
    effect    = "Allow"
    actions   = ["ec2:DescribeManagedPrefixLists", "ec2:GetManagedPrefixListEntries"]
    resources = ["*"]
  }
  # RDS
  statement {
    sid       = "ColorTokensRDSReadAccess"
    effect    = "Allow"
    actions   = ["rds:DescribeDBInstances", "rds:DescribeDBClusters", "rds:DescribeDBClusterEndpoints", "rds:DescribeDBSecurityGroups", "rds:DescribeDBSubnetGroups", "rds:DescribeDBParameterGroups", "rds:ListTagsForResource"]
    resources = ["*"]
  }
  # Elastic Load Balancing
  statement {
    sid       = "ColorTokensELBReadAccess"
    effect    = "Allow"
    actions   = ["elasticloadbalancing:DescribeLoadBalancers", "elasticloadbalancing:DescribeTags", "elasticloadbalancing:DescribeTargetGroups", "elasticloadbalancing:DescribeListeners", "elasticloadbalancing:DescribeTargetHealth"]
    resources = ["*"]
  }
  # API Gateway
  statement {
    sid       = "ColorTokensAPIGatewayReadAccess"
    effect    = "Allow"
    actions   = ["apigateway:GET"]
    resources = ["*"]
  }
  # X-Ray
  statement {
    sid       = "ColorTokensXRayReadAccess"
    effect    = "Allow"
    actions   = ["xray:GetRetrievedTracesGraph", "xray:GetGroups", "xray:GetSamplingStatisticSummaries", "xray:CancelTraceRetrieval", "xray:GetTraceGraph", "xray:GetServiceGraph", "xray:GetInsightImpactGraph", "xray:GetTraceSegmentDestination", "xray:GetInsightSummaries", "xray:GetSamplingTargets", "xray:GetGroup", "xray:BatchGetTraceSummaryById", "xray:StartTraceRetrieval", "xray:GetTimeSeriesServiceStatistics", "xray:GetEncryptionConfig", "xray:GetSamplingRules", "xray:GetInsight", "xray:GetDistinctTraceGraphs", "xray:GetInsightEvents", "xray:GetTraceSummaries", "xray:GetIndexingRules", "xray:BatchGetTraces", "xray:ListResourcePolicies", "xray:ListRetrievedTraces", "xray:ListTagsForResource"]
    resources = ["*"]
  }
}
